// MediaChanger.cpp : Defines the entry point for the console application.
//

#include "stdafx.h"
#include "ChangerControl.h"
#include <winioctl.h>

void Usage()
{
	printf("usage:\n");
	printf("  ChangerControl mount <slot number>\n");
	printf("     mount the disk in slot <slot number> to the drive\n");
	printf("  ChangerControl unmount [<slot number>]\n");
	printf("     move the disk in the drive to slot <slot number>.  If <slot number> is not\n");
	printf("     specified then put it back to the slot that it came from.\n");
	printf("  ChangerControl slotfull [<slot number>]\n");
	printf("     Is there a disk in slot <slot number>?  If <slot number> is not specified\n");
	printf("     then display the status of all slots\n");
	printf("  ChangerControl drivestatus\n");
	printf("     Show the status of the drive.  Returns -2 if drive is empty, -1 on error, \n");
	printf("     source otherwise.\n");
	printf("  ChangerControl fullslots\n");
	printf("     List all slots with media\n");
	printf("  ChangerControl emptyslots\n");
	printf("     List all slots without media\n");
	return;
}

int __cdecl main(int argc, char **argv)
/*++

Routine Description:

    This is the main function. It takes no arguments from the user.

Arguments:

    None

Return Value:

  Status

--*/
{
	if (argc == 1)
	{
		Usage();
		return -1;
	}

	CChangerControl *pChanger = new CChangerControl();
	HRESULT hr;

	hr = pChanger->Init();
	if (FAILED(hr))
	{
		printf("CChangerControl.Init failed with 0x%x (do you have a changer?  Is it turned on?)", hr);
	}

	if (stricmp(argv[1], "mount") == 0)
	{
		if (argc != 3) 
		{
			Usage(); 
			return -1;
		}
		// get the slot number
  		DWORD slotNumber = atoi(argv[2]);

		// BUGBUG -- We should ask the changer how many slots it has
		if (slotNumber < 0 || slotNumber > 199)
		{
			printf("slot number is invalid\n");
			return -1;
		}

		// see if the drive has a disk already.  If so error out.
		HRESULT hr = pChanger->GetDriveStatus();
		if (FAILED(hr))
		{
			printf("IsElementFull(ChangerDrive, 0) failed with 0x%x\n", hr);
			return -1;
		}
		if (hr == S_OK)
		{
			printf("The drive already contains a disk, use ChangerControl unmount first\n");
			return -1;
		}

		// Move the disk to the drive
		hr = pChanger->MoveMedium(ChangerSlot, slotNumber, ChangerDrive, 0);
		if (FAILED(hr))
		{
			printf("MoveMedium(ChangerSlot, %i, ChangerDrive, 0) failed with 0x%x\n", slotNumber, hr);
			return -1;
		}
		return 0;
	}
	else if (stricmp(argv[1], "unmount") == 0)
	{
		int slotNumber;
		BOOL fFull;
		// get the slot that this media came from
		hr = pChanger->GetDriveStatus(&fFull, &slotNumber);
		if (FAILED(hr))
		{
			printf("GetDriveStatus() failed with 0x%x\n", hr);
			return -1;
		}

		if (!fFull)
		{
			printf("The drive is empty, use ChangerControl mount first\n");
			return -1;
		}

		if (argc == 3) 
		{
			// get the slot number from the command line
	  		slotNumber = atoi(argv[2]);
			// BUGBUG -- We should ask the changer how many slots it has
			if (slotNumber < 0 || slotNumber > 199)
			{
				printf("slot number is invalid\n");
				return -1;
			}
		}
		else if (argc != 2)
		{
			Usage(); 
			return -1;
		}

		// Move the disk to the slot
		hr = pChanger->MoveMedium(ChangerDrive, 0, ChangerSlot, slotNumber);
		if (FAILED(hr))
		{
			printf("MoveMedium(ChangerDrive, 0, ChangerSlot, %i) failed with 0x%x\n", slotNumber, hr);
			return -1;
		}
		return 0;
	}
	else if (stricmp(argv[1], "slotfull") == 0)
	{
		DWORD slotNumber;
		if (argc == 2) 
		{
			// dump all slots
			for (slotNumber = 0; 
				 slotNumber < 200; 
				 slotNumber++)
			{
				hr = pChanger->IsSlotFull(slotNumber);
				if (FAILED(hr))
				{
					printf("IsSlotFull(%i) failed with 0x%x\n", slotNumber, hr);
					return -1;
				}
				printf("%03i: %s\n", slotNumber, (hr == S_OK) ? "full" : "empty");
			}
			return 0;
		}
		else if (argc == 3)
		{
			// get the slot number from the command line
	  		slotNumber = atoi(argv[2]);
			// BUGBUG -- We should ask the changer how many slots it has
			if (slotNumber < 0 || slotNumber > 199)
			{
				printf("slot number is invalid\n");
				return -1;
			}
			hr = pChanger->IsSlotFull(slotNumber);
			if (FAILED(hr))
			{
				printf("IsSlotFull(%i) failed with 0x%x", slotNumber, hr);
				return -1;
			}
			printf("%03i: %s\n", slotNumber, (hr == S_OK) ? "full" : "empty");
			return (hr == S_OK) ? 1 : 0;
		}
		else
		{
			Usage(); 
			return -1;
		}
	}
	else if (stricmp(argv[1], "drivestatus") == 0)
	{
		if (argc == 2)
		{
			BOOL fFull;
			int iSourceAddress;
			hr = pChanger->GetDriveStatus(&fFull, &iSourceAddress);
			if (FAILED(hr))
			{
				printf("IsElementFull(ChangerDrive, 0) failed with 0x%x", hr);
				return -1;
			}
			printf("drive: %s, source=%i\n", fFull ? "full" : "empty", iSourceAddress);
			return (hr == S_OK) ? iSourceAddress : -2;
		}
		else
		{
			Usage(); 
			return -1;
		}
	}
	else if (stricmp(argv[1], "fullslots") == 0 || 
			 stricmp(argv[1], "emptyslots") == 0)
	{
		BOOL fListFull = (stricmp(argv[1], "fullslots") == 0);
		// dump all slots
		for (int slotNumber = 0; slotNumber < 200; slotNumber++)
		{
			hr = pChanger->IsSlotFull(slotNumber);
			if (FAILED(hr))
			{
				printf("IsSlotFull(%i) failed with 0x%x\n", slotNumber, hr);
				return -1;
			}
			if (hr == S_OK && fListFull)
			{
				printf("%03i: full\n", slotNumber);
			}
			else if (hr == S_FALSE && !fListFull)
			{
				printf("%03i: empty\n", slotNumber);
			}
		}
		return 0;
	}
	else if (stricmp(argv[1], "next") == 0)
	{
		// mount the next disk after this one.  If we are at disk 199 then give
		// up.  If nothing is mounted then give up.
		BOOL fFull;
		int iSourceAddress;
		hr = pChanger->GetDriveStatus(&fFull, &iSourceAddress);
		if (FAILED(hr))
		{
			printf("IsElementFull(ChangerDrive, 0) failed with 0x%x", hr);
			return -1;
		}

		if (!fFull)
		{
			printf("MediaChanger next requires a disk to be mounted.\n");
			return -1;
		}

		// find the next slot that has a free disk
		int iNextSlot = -1;
		for (int i = iSourceAddress + 1; i < 199 && iNextSlot == -1; i++)
		{
			hr = pChanger->IsSlotFull(i);
			if (FAILED(hr))
			{
				printf("IsSlotFull(%i) returned 0x%x\n", i, hr);
				return -1;
			}
			if (hr == S_OK) iNextSlot = i;
		}
		if (iNextSlot == -1)
		{
			printf("Nothing left to mount!");
			return 0;
		}

		// Move the disk to the slot
		printf("disk in drive to slot %i\n", iSourceAddress);
		hr = pChanger->MoveMedium(ChangerDrive, 0, ChangerSlot, iSourceAddress);
		if (FAILED(hr))
		{
			printf("MoveMedium(ChangerDrive, 0, ChangerSlot, %i) failed with 0x%x\n", iSourceAddress, hr);
			return -1;
		}

		// Move the disk to the slot
		printf("disk in slot %i to drive\n", iNextSlot);
		hr = pChanger->MoveMedium(ChangerSlot, iNextSlot, ChangerDrive, 0);
		if (FAILED(hr))
		{
			printf("MoveMedium(ChangerSlot, %i, ChangerDrive, 0) failed with 0x%x\n", iNextSlot, hr);
			return -1;
		}

		return 1;
	}
	else
	{
		Usage();
		return -1;
	}

	delete pChanger;
}


